/******************************************************************************/
/*** Preamble *****************************************************************/
/******************************************************************************/

/*
This is a simple offline judgement test. 

Participants read a sentence and answer a comprehensive question about the sentence.

The question is shown in a multi-choice way. The order of the questions and the options 
are auto-randomized.

There are four conditions in the experiment, which means there are four lists. 
Participants are randomly assigned to the four list. 


/******************************************************************************/
/*** Initialise jspsych *******************************************************/
/******************************************************************************/

/*
The progress bar is defined here.
*/

var jsPsych = initJsPsych({
  show_progress_bar: true,
  message_progress_bar: '',
  default_iti: 200,
  on_finish: function () {
    //jsPsych.data.displayData("csv"); //dump the data to screen
  },
});



/******************************************************************************/
/*** Saving data trial by trial ***********************************************/
/******************************************************************************/

/*
This is the save_data function provided in Alisdair's tutorial, section 06. 
*/
function save_data(name, data_in) {
  var url = "save_data.php";
  var data_to_send = { filename: name, filedata: data_in };
  fetch(url, {
    method: "POST",
    body: JSON.stringify(data_to_send),
    headers: new Headers({
      "Content-Type": "application/json",
    }),
  });
}

function save_pronoun_production_data(data) {
  // choose the data we want to save - this will also determine the order of the columns
  if (data.participant_task == "pronoun_production") {
    var data_to_save = [
      participant_id,
      data.trial_index,
      data.participant_task,
      data.ori_index,
      data.type,
      data.verb,
      data.IC_bias,
      data.IR_bias,
      data.IC_list,
      data.IR_list,
      data.connective,
      data.sentence,
      data.response.response,
      data.rt,
    ];
  } 
  // join these with commas and add a newline
  var line = data_to_save.join(",") + "\n";
  var this_participant_filename = "pp_" + participant_id + ".csv";
  save_data(this_participant_filename, line);
}


function save_comprehension_data(data) {
  // choose the data we want to save - this will also determine the order of the columns
  if (data.participant_task == "comprehension") {
    var data_to_save = [
      participant_id,
      data.trial_index,
      data.participant_task,
      data.ori_index,
      data.sentence,
      data.question,
      data.answer,
      data.response,
      data.rt,
    ];
  } 
  // join these with commas and add a newline
  var line = data_to_save.join(",") + "\n";
  var this_participant_filename = "com_" + participant_id + ".csv";
  save_data(this_participant_filename, line);
}
/******************************************************************************/
/*** Generate a random participant ID *****************************************/
/******************************************************************************/

/*
We'll generate a random participant ID when the experiment starts, and use this
to save a seperate set of data files per participant.
*/

var participant_id = jsPsych.randomization.randomID(10);


/******************************************************************************/
/*** pronoun production trials *************************************************/
/******************************************************************************/


function make_pronoun_production_trial(ori_index,type,verb,IC_bias,IR_bias,IC_list,IR_list,connective,sentence,question,answer){
  var selection_trial = {
    type: jsPsychSurveyHtmlForm,
    html: '<p>' + sentence + '<input type="text" id="test-resp-box" name="response" required size="40" /> </p>', 
    button_label: 'Continue',
    autofocus: 'test-resp-box',
    on_start: function (trial) {
      trial.data = {
        participant_task: "pronoun_production",
        ori_index: ori_index,
        type: type, 
        verb: verb,
        IC_bias: IC_bias,
        IR_bias: IR_bias,
        IC_list: IC_list,
        IR_list: IR_list,
        connective: connective,
        sentence: sentence,
      };
    },
    on_finish: function (data) {
      //var res = data.response.response;
      //console.log(res);
      save_pronoun_production_data(data); //save the trial data
    },
  }

  var comprehension_trial = {
    type: jsPsychHtmlButtonResponse,
    stimulus: question,
    choices: ['Yes', 'No'],
    prompt: 'Does the statement above matched the last sentence you saw?',
    on_start: function (trial) {
      trial.data = {
        participant_task: "comprehension",
        ori_index: ori_index,
        sentence: sentence,
        question: question,
        answer: answer
      };
    },
    on_finish:function (data) {
      save_comprehension_data(data);
    }
  }

  var full_trial = {timeline: []};

  if (question == 'skip'){
    full_trial = {timeline: [selection_trial]};
  } else {
    full_trial = {timeline: [selection_trial, comprehension_trial]};
  }

  return full_trial;
}




/******************************************************************************/
/*** Exercise trials *************************************************/
/******************************************************************************/

var exercise_list = [
  {target: 'Xiaoming finally saw Xiaoqiang yesterday, but', question: 'skip'},
  {target: 'Xiaomei prepared a birthday party for Xiaoli. Suddenly', question: 'skip'},
  {target: 'Mr Wang did not see Xiaozhang in the office. Maybe', question: 'Mr Wang has never been to the office.'}
];


function make_exercise_trial(target, question) {

  var exercise_trial = {
    type: jsPsychSurveyHtmlForm,
    preamble: "<p style='color:red'> <i> *** The sentences contain interactions between <b>「two characters」</b>.***</i></p>\
             <p style='color:red'> <i>***Please use <b>「one of the characters」</b> to provide a possible continuation for the sentence.*** </i></p>\
             <p style='color:red''> <i>***When you finish, click on <b>「Continue」</b> or press <b>「Enter」</b> to move to the next sentence.***</i></p>",
    html: '<p>' + target + '<input type="text" required id="test-resp-box" name="response" size="40" /> </p>', 
    autofocus: 'test-resp-box',
    button_label: 'Continue',
  };


  var exercise_trial2 = {
    type: jsPsychHtmlButtonResponse,
    stimulus: "<p style='color:red'> <i> *** After some of the sentences, you may see a <b>comprehension question </b> ***</i></p>\
    <p style='color:red'> <i> If you think the statement match the last sentence you have just read, click on 'Yes', otherwise, click on 'No'***</i></p>" 
    + question,
    choices: ['Yes', 'No'],
    prompt: 'Does the statement above matched the last sentence you have just seen?',
  }

  var full_trial = {timeline: []};

  if (question == 'skip'){
    full_trial = {timeline: [exercise_trial]};
  } else {
    full_trial = {timeline: [exercise_trial, exercise_trial2]};
  }


  return full_trial;
}

/******************************************************************************/
/*** Example trials *************************************************/
/******************************************************************************/

var example_list = [
  {target: 'John should have met Jack today, but ', fill: 'he/forgot/the/meeting.', question: 'skip'},
  {target: 'Amy did not see Tammy today. Then ', fill: 'she/called/Tammy/by/her/phone.', question: 'skip'},
  {target: 'Mary taught Anna how to make a cake. However, ', fill: 'Anna/still/finds/it/very/difficult.',
    question: 'Anna tried to learn how to make a cake from Mary.'}
];


function make_example_trial(target, fill, question) {

  var example_trial = {
    type: jsPsychSurveyHtmlForm,
    preamble: "<p style='color:red'> <i> *** The sentences contain interactions between <b>「two characters」</b>.***</i></p>\
             <p style='color:red'> <i>***Please use <b>「one of the characters」</b> to provide a possible continuation for the sentence.*** </i></p>\
             <p style='color:red''> <i>***When you finish, click on <b>「Continue」</b> or press <b>「Enter」</b> to move to the next sentence.***</i></p>",
    html: '<p>' + target + '<input type="text" required id="test-resp-box" size="40" value=' + fill + '> </p>', 
    button_label: 'Continue',
  };


  var example_trial2 = {
    type: jsPsychHtmlButtonResponse,
    stimulus: "<p style='color:red'> <i> *** After some of the sentences, you may see a <b>comprehension question </b> ***</i></p>\
    <p style='color:red'> <i>***If you think the statement match the last sentence you have just read, click on 'Yes', otherwise, click on 'No'.***</i></p>"
    + question,
    choices: ['Yes', 'No'],
    prompt: 'Does the statement above matched the last sentence you have just seen?',
  }

  var full_trial = {timeline: []};

  if (question == 'skip'){
    full_trial = {timeline: [example_trial]};
  } else {
    full_trial = {timeline: [example_trial, example_trial2]};
  }

  return full_trial;
}


/******************************************************************************/
/*** Rest block *************************************************/
/******************************************************************************/

function make_rest_block (video, sec_number){
  var rest_block = {
    type: jsPsychVideoButtonResponse,
    stimulus: [video],
    choices: [],
    prompt: "<p> Now we can take a rest.</p>",
    autoplay: true,
    trial_duration: 12000,
    width: '525px',
  };

  var after_rest = {
    type: jsPsychHtmlButtonResponse,
    stimulus: "<h3> Let's continue!</h3>\
    <p> We still have " + sec_number + " sections to go. </p>",
    choices: ['Continue']
  }

  var full_trial = {timeline: [rest_block, after_rest]};

  return full_trial
}

/******************************************************************************/
/*** Write headers for data file **********************************************/
/******************************************************************************/


var write_pp_headers = {
  type: jsPsychCallFunction,
  func: function () {
    var this_participant_filename = "pp_" + participant_id + ".csv";
    save_data(
      this_participant_filename,
      "participant_id,trial_index,participant_task,ori_index,type,verb,IC_bias,IR_bias,\
      IC_list, IR_list, connective, sentence,response, rt\n"
    );
  },
};

var write_com_headers = {
  type: jsPsychCallFunction,
  func: function () {
    var this_participant_filename = "com_" + participant_id + ".csv";
    save_data(
      this_participant_filename,
      "participant_id,trial_index,participant_task,ori_index,sentence,question,answer,response,rt\n"
    );
  },
};

/******************************************************************************/
/*** info screens *******************************************************/
/******************************************************************************/



var pre_consent_sreen = {
  type: jsPsychHtmlButtonResponse,
  stimulus:
    " <h3>Welcome to the experiment! </h3> \
    <p style='text-align:left; padding-left:20%; padding-right:20%'>You are invited to participate in a task about how people understand and produce sentences. </p>\
    <p style='text-align:left; padding-left:20%; padding-right:20%'>Before you start the task, we will ask for your consent of participation. \
    On the next page, we will show you information about your risks and benefits, your compensation, \
    your right to withdraw and how we will use and protect your data. </p>\
    <p style='text-align:left; padding-left:20%; padding-right:20%'>After your consent, you will get full instructions about how to participate in the task and how to claim your compensation. </p>\
    <p style='text-align:left; padding-left:20%; padding-right:20%'> Click on 'Continue' to move to the consent page. </p>",
  choices: ["Continue"],
};



var consent_screen = {
  type: jsPsychHtmlButtonResponse,
  stimulus:
  "<h3> Consent of participation </h3> \
  <p style='text-align:left'><b>Project title:</b> Chinese Sentence Completion Study <br>\
  <b>Principal Investigator:</b> Professor Vicky Chondrogianni; Dr Patrick Sturt <br>\
  <b>Researcher collecting data:</b> Shuya Chen</p> \
  <p style='text-align:left'><b>About the study. </b> \
  This study is about how people understand and produce sentences.\
  Your responses will be recorded. Your session should last for up to <b>30 minutes</b>. You will be given full instructions shortly.</p>\
  <p style='text-align:left'><b>Compensation. </b> You will be paid ____RMB20___for your participation in this study. </p>\
  <p style='text-align:left'><b>Voluntary participation and right to withdraw. </b> \
   We will ask your permission before we begin the study, and we will make sure you understand that you can stop the study at any point. \
   You may also choose to withdraw from the study until two weeks after participating in the study. \
   Any data supplied up to that point will be deleted. Please note that if you decide to withdraw after the specified date, we may \
   not be able to delete your anonymised data from e.g. public repositories of research data, but we will be able to delete \
   all of your personal data from our records.</p>\
  <p style='text-align:left'><b>Risks and benefits. </b>\
   There are no known risks to participating in this study. \
   There are no tangible benefits to you, however, you will be contributing to our knowledge about language processing and language acquisition.</p>\
  <p style='text-align:left'><b>Confidentiality and use of data. </b> \
   All the information we collect during the course of the research will be processed in accordance \
   with the Data Protection Law. In order to safeguard your own privacy, we will not collect any personal information. \
   Your data will be only referred to by a unique participant number. \
   The anonymised data collected during this study will be used for research purposes and may be shared with other researchers.</p>\
  <p style='text-align:left'><b>What are my data protection rights? </b> \
   The University of Edinburgh is a Data Controller for the information you provide. \
   You have the right to access relevant information held about you. \
   Your right of access can be exercised in accordance with the Data Protection Law. You also have other rights \
   including rights of correction, erasure and objection.  For more details, including the right to lodge a \
   complaint with the Information Commissioner’s Office, please visit www.ico.org.uk.  Questions, comments and \
   requests about your personal data can also be sent to the University Data Protection Officer at dpo@ed.ac.uk. </p>\
  <p style='text-align:left'>If you have any questions about what you’ve just read, please feel free to ask, or contact us later. \
  You can contact us by email at s2164197@ed.ac.uk. This project has been approved by the PPLS Ethics Committee. \
  If you have questions or comments regarding your rights as a participant, they can be contacted at \
  0131 650 4020 or ppls.ethics@ed.ac.uk. </p>\
  <p style='text-align:left'>By clicking on <b>‘I agree’</b>, you consent to the following: <br>\
  1.	<b>I consent to taking part in the above study.</b><br>\
  2.	I confirm that I have read and understood <b>how my data will be stored and used.</b><br>\
  3.	I understand that I have the <b>right to terminate this session</b> until two weeks after participating in the study. \
  If I choose to <b>withdraw after completing the study</b>, my data will be deleted at that time.</p>\ ",
  choices: ["I agree"],
  margin_vertical: '24px'
};


/******************************************************************************/
/*** instructions *******************************************************/
/******************************************************************************/

var instruction_screen_prepractise_1 = {
  type: jsPsychHtmlButtonResponse,
  stimulus:
    "<p style='text-align:left; color:red'> <i> <b> Please read the instructions carefully. </b> </i> </p>\
  <p style='text-align:left'>In the following parts, you will see a set of <b> short sentences </b>.</p>\
  <p style='text-align:left'>You will need to provide a <b>possible continuation</b> for each of the sentence.</p>",
  choices: ["Continue"],
};

var instruction_screen_prepractise_2 = {
  type: jsPsychHtmlButtonResponse,
  stimulus:
    "<p style='text-align:left; color:red'> <i> <b> Please read the instructions carefully. </b> </i> </p>\
  <p style='text-align:left'> The sentences contain interactions between <b>「two characters」</b> </p>\
  <p style='text-align:left'> Please use <b>「one of the characters」 </b> to provide a possible continuation for the sentence. </p>\
  <p style='text-align:left'> When you finish, click on <b>「Continue」</b> or press <b>「Enter/Return」</b> to move to the next sentence.</b></p>" ,
  choices: ["Continue"],
};


var instruction_screen_prepractise_3 = {
  type: jsPsychHtmlButtonResponse,
  stimulus:
    "<p style='text-align:left; color:red'> <i> <b> Please read the instructions carefully. </b> </i> </p>\
  <p style='text-align:left'> The whole experiment will take you no more than <b>30 minutes</b> to complete.</p> \
  <p style='text-align:left'>We will ask you to take some <b>short breaks</b> throughout the experiment.</p>\
  <p style='text-align:left'>During the experiment, you will be able to see the progress on the <b>progress bar</b> on the top of the screen. </p>" ,
  choices: ["Continue"],
};



var instruction_screen_prepractise_4 = {
  type: jsPsychHtmlButtonResponse,
  stimulus:
    "<p style='text-align:left; color:red'> <i> <b> Please read the instructions carefully. </b> </i> </p>\
  <p style='text-align:left'> When you finish the experiment, you will see a <b>completion code</b>. </p>\
  <p style='text-align:left'>Please save your completion code for compensation claiming. </p>" ,
  choices: ["Continue"],
};

var instruction_screen_prepractise_5 = {
  type: jsPsychHtmlButtonResponse,
  stimulus:
    "<p> <b> Now let's see some examples!</b> </p>\
  <p style='text-align:left'> You will see three examples with <b>prefilled answers</b>.</p>\
  <p style='text-align:left'> Click on 'Continue' to move on to the examples.</p>" ,
  choices: ["Continue"],
};


var instruction_screen_prepractise = [
  instruction_screen_prepractise_1,
  instruction_screen_prepractise_2,
  instruction_screen_prepractise_3,
  instruction_screen_prepractise_4,
  instruction_screen_prepractise_5
]

var instruction_screen_btwpractise = {
  type: jsPsychHtmlButtonResponse,
  stimulus:
    "<p> <b> Now let's practise!</b> </p>\
  <p style='text-align:left'> There are three sentences in the practise section.</p>\
  <p style='text-align:left'> Click on 'Continue' to move on to the practise section.</p>" ,
  choices: ["Continue"],
};


/******************************************************************************/
/*** post practise *******************************************************/
/******************************************************************************/


var instruction_screen_postpractise = {
  type: jsPsychHtmlButtonResponse,
  stimulus:
    "<p>You have finished all the practise!</p>\
    <p> <b>Now let's move on to the experiment.</b></p>",
  choices: ["Start the experiment"],
};


var final_screen = {
  type: jsPsychHtmlButtonResponse,
  stimulus:
    "<h3> You've finished the last task. Thanks for participating! </h3>\
  <p style='text-align:left'>Your completion code is <b>" + participant_id + "</b></p>" +
  "<p style='text-align:left'> Please copy the completion code and send it to the researcher for compensation claiming. You will receive your payment within one week.</p>\
  <p style='text-align:left'>If you have any questions, please contact the researcher (Shuya Chen, s2164197@ed.ac.uk)。</p>",
  choices: ["Finish"],
};

var after_final = {
  type: jsPsychVideoButtonResponse,
  stimulus: [
    'video/penguin.mp4'
  ],
  choices: [],
  autoplay: true,
  trial_duration: 12000,
  width: '525px',
};


/******************************************************************************/
/*** Reading the trial list from a CSV file ***********************************/
/******************************************************************************/

/*
This function reads the trial list provided in triallist_filename (using code in a 
separate file, read_from_csv.js), and converts it to a series of jsPsych trials. 
We read the trial list in as a javascript array using read_trial_list (defined 
in read_from_csv.js). Since reading the CSV file takes some time, we have to 
use the async and await functions to ensure that the CSV file has been read before we 
start processing it. We then use build_timeline to turn the array into a list of jsPsych 
trials. 
*/

async function read_trials_and_prepare_timeline(triallist_filename) {
  var trial_list = await read_trial_list(triallist_filename);
  var production_trials = build_timeline(trial_list);
  //console.log(trial_list);
  var exercise_trials = build_exercise(exercise_list);
  var example_trials = build_example(example_list);
  //console.log(exercise_list);
  var full_timeline = [].concat(
    //preload,
    pre_consent_sreen,
    consent_screen,
    write_pp_headers,
    write_com_headers,
    instruction_screen_prepractise,
    example_trials,
    instruction_screen_btwpractise,
    exercise_trials,
    instruction_screen_postpractise,
    production_trials,
    final_screen,
    after_final
  );
  jsPsych.run(full_timeline);
}

/*
build_timeline takes a trial list read from a CSV and uses make_sentence_judgement_trial
to convert each row of that CSV file into a jsPsych trial, 
reading the relevant info from the appropriate columns in the CSV data structure.
*/
function build_timeline(trial_list) {
  var production_trials = []; 
  var trial_list_ran = jsPsych.randomization.shuffle(trial_list);
  var trial_number = 0;
  for (trial of trial_list_ran) {
      trial_number = trial_number + 1;
      var match_trial = make_pronoun_production_trial(
        trial.ori_index,
        trial.type,
        trial.verb,
        trial.IC_bias,
        trial.IR_bias,
        trial.IC_list,
        trial.IR_list,
        trial.connective,
        trial.sentence,
        trial.question,
        trial.answer
      );

      //var rest_trial = make_rest_block(video);
      
      if (trial_number == 20){
        production_trials.push(match_trial);
        production_trials.push(make_rest_block('video/cat1.mp4', "三"));
      } else if (trial_number == 40) {
        production_trials.push(match_trial);
        production_trials.push(make_rest_block('video/cat2.mp4', "二"));
      } else if (trial_number == 60) {
        production_trials.push(match_trial);
        production_trials.push(make_rest_block('video/cat3.mp4', "一"));
      } else {
        production_trials.push(match_trial);
      }
  }
  return production_trials;
}

function build_exercise(trial_list) {
  var exercise_trials = [];
  for (trial of trial_list) {
      var match_trial = make_exercise_trial(
        trial.target,
        trial.question
      );
      exercise_trials.push(match_trial);
  }
  return exercise_trials;
}

function build_example(trial_list) {
  var example_trials = [];
  for (trial of trial_list) {
      var match_trial = make_example_trial(
        trial.target,
        trial.fill,
        trial.question
      );
      example_trials.push(match_trial);
  }
  return example_trials;
}

/*
run everything here

read_trials_and_prepare_timeline("sentence_judgement_A.csv");
*/


/*
randomly assigning participants into four lists
*/
function random_condition() {
  var available_csvs = [
    "pronoun_production_A.csv"
  ];
  var selected_csv = jsPsych.randomization.shuffle(available_csvs)[0];
  return selected_csv;
}
var this_condition = random_condition();

/*add the list with the participant id so that I can see which list they are assigned in*/
var list = this_condition.slice(18,20);
var participant_id = participant_id + list;

read_trials_and_prepare_timeline(this_condition);



  